/*! firebase-admin v10.3.0 */
"use strict";
/*!
 * @license
 * Copyright 2021 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirebaseFunctionsError = exports.FUNCTIONS_ERROR_CODE_MAPPING = exports.FunctionsApiClient = void 0;
var api_request_1 = require("../utils/api-request");
var error_1 = require("../utils/error");
var utils = require("../utils/index");
var validator = require("../utils/validator");
var CLOUD_TASKS_API_URL_FORMAT = 'https://cloudtasks.googleapis.com/v2/projects/{projectId}/locations/{locationId}/queues/{resourceId}/tasks';
var FIREBASE_FUNCTION_URL_FORMAT = 'https://{locationId}-{projectId}.cloudfunctions.net/{resourceId}';
var FIREBASE_FUNCTIONS_CONFIG_HEADERS = {
    'X-Firebase-Client': "fire-admin-node/" + utils.getSdkVersion()
};
// Default canonical location ID of the task queue.
var DEFAULT_LOCATION = 'us-central1';
/**
 * Class that facilitates sending requests to the Firebase Functions backend API.
 *
 * @internal
 */
var FunctionsApiClient = /** @class */ (function () {
    function FunctionsApiClient(app) {
        this.app = app;
        if (!validator.isNonNullObject(app) || !('options' in app)) {
            throw new FirebaseFunctionsError('invalid-argument', 'First argument passed to getFunctions() must be a valid Firebase app instance.');
        }
        this.httpClient = new api_request_1.AuthorizedHttpClient(app);
    }
    /**
     * Creates a task and adds it to a queue.
     *
     * @param data - The data payload of the task.
     * @param functionName - The functionName of the queue.
     * @param extensionId - Optional canonical ID of the extension.
     * @param opts - Optional options when enqueuing a new task.
     */
    FunctionsApiClient.prototype.enqueue = function (data, functionName, extensionId, opts) {
        var _this = this;
        if (!validator.isNonEmptyString(functionName)) {
            throw new FirebaseFunctionsError('invalid-argument', 'Function name must be a non empty string');
        }
        var task = this.validateTaskOptions(data, opts);
        var resources;
        try {
            resources = utils.parseResourceName(functionName, 'functions');
        }
        catch (err) {
            throw new FirebaseFunctionsError('invalid-argument', 'Function name must be a single string or a qualified resource name');
        }
        if (typeof extensionId !== 'undefined' && validator.isNonEmptyString(extensionId)) {
            resources.resourceId = "ext-" + extensionId + "-" + resources.resourceId;
        }
        return this.getUrl(resources, CLOUD_TASKS_API_URL_FORMAT)
            .then(function (serviceUrl) {
            return _this.updateTaskPayload(task, resources)
                .then(function (task) {
                var request = {
                    method: 'POST',
                    url: serviceUrl,
                    headers: FIREBASE_FUNCTIONS_CONFIG_HEADERS,
                    data: {
                        task: task,
                    }
                };
                return _this.httpClient.send(request);
            });
        })
            .then(function () {
            return;
        })
            .catch(function (err) {
            throw _this.toFirebaseError(err);
        });
    };
    FunctionsApiClient.prototype.getUrl = function (resourceName, urlFormat) {
        var _this = this;
        var locationId = resourceName.locationId;
        var projectId = resourceName.projectId, resourceId = resourceName.resourceId;
        if (typeof locationId === 'undefined' || !validator.isNonEmptyString(locationId)) {
            locationId = DEFAULT_LOCATION;
        }
        return Promise.resolve()
            .then(function () {
            if (typeof projectId !== 'undefined' && validator.isNonEmptyString(projectId)) {
                return projectId;
            }
            return _this.getProjectId();
        })
            .then(function (projectId) {
            var urlParams = {
                projectId: projectId,
                locationId: locationId,
                resourceId: resourceId,
            };
            // Formats a string of form 'project/{projectId}/{api}' and replaces
            // with corresponding arguments {projectId: '1234', api: 'resource'}
            // and returns output: 'project/1234/resource'.
            return utils.formatString(urlFormat, urlParams);
        });
    };
    FunctionsApiClient.prototype.getProjectId = function () {
        var _this = this;
        if (this.projectId) {
            return Promise.resolve(this.projectId);
        }
        return utils.findProjectId(this.app)
            .then(function (projectId) {
            if (!validator.isNonEmptyString(projectId)) {
                throw new FirebaseFunctionsError('unknown-error', 'Failed to determine project ID. Initialize the '
                    + 'SDK with service account credentials or set project ID as an app option. '
                    + 'Alternatively, set the GOOGLE_CLOUD_PROJECT environment variable.');
            }
            _this.projectId = projectId;
            return projectId;
        });
    };
    FunctionsApiClient.prototype.getServiceAccount = function () {
        var _this = this;
        if (this.accountId) {
            return Promise.resolve(this.accountId);
        }
        return utils.findServiceAccountEmail(this.app)
            .then(function (accountId) {
            if (!validator.isNonEmptyString(accountId)) {
                throw new FirebaseFunctionsError('unknown-error', 'Failed to determine service account. Initialize the '
                    + 'SDK with service account credentials or set service account ID as an app option.');
            }
            _this.accountId = accountId;
            return accountId;
        });
    };
    FunctionsApiClient.prototype.validateTaskOptions = function (data, opts) {
        var task = {
            httpRequest: {
                url: '',
                oidcToken: {
                    serviceAccountEmail: '',
                },
                body: Buffer.from(JSON.stringify({ data: data })).toString('base64'),
                headers: { 'Content-Type': 'application/json' }
            }
        };
        if (typeof opts !== 'undefined') {
            if (!validator.isNonNullObject(opts)) {
                throw new FirebaseFunctionsError('invalid-argument', 'TaskOptions must be a non-null object');
            }
            if ('scheduleTime' in opts && 'scheduleDelaySeconds' in opts) {
                throw new FirebaseFunctionsError('invalid-argument', 'Both scheduleTime and scheduleDelaySeconds are provided. '
                    + 'Only one value should be set.');
            }
            if ('scheduleTime' in opts && typeof opts.scheduleTime !== 'undefined') {
                if (!(opts.scheduleTime instanceof Date)) {
                    throw new FirebaseFunctionsError('invalid-argument', 'scheduleTime must be a valid Date object.');
                }
                task.scheduleTime = opts.scheduleTime.toISOString();
            }
            if ('scheduleDelaySeconds' in opts && typeof opts.scheduleDelaySeconds !== 'undefined') {
                if (!validator.isNumber(opts.scheduleDelaySeconds) || opts.scheduleDelaySeconds < 0) {
                    throw new FirebaseFunctionsError('invalid-argument', 'scheduleDelaySeconds must be a non-negative duration in seconds.');
                }
                var date = new Date();
                date.setSeconds(date.getSeconds() + opts.scheduleDelaySeconds);
                task.scheduleTime = date.toISOString();
            }
            if (typeof opts.dispatchDeadlineSeconds !== 'undefined') {
                if (!validator.isNumber(opts.dispatchDeadlineSeconds) || opts.dispatchDeadlineSeconds < 15
                    || opts.dispatchDeadlineSeconds > 1800) {
                    throw new FirebaseFunctionsError('invalid-argument', 'dispatchDeadlineSeconds must be a non-negative duration in seconds '
                        + 'and must be in the range of 15s to 30 mins.');
                }
                task.dispatchDeadline = opts.dispatchDeadlineSeconds + "s";
            }
            if (typeof opts.uri !== 'undefined') {
                if (!validator.isURL(opts.uri)) {
                    throw new FirebaseFunctionsError('invalid-argument', 'uri must be a valid URL string.');
                }
                task.httpRequest.url = opts.uri;
            }
        }
        return task;
    };
    FunctionsApiClient.prototype.updateTaskPayload = function (task, resources) {
        var _this = this;
        return Promise.resolve()
            .then(function () {
            if (validator.isNonEmptyString(task.httpRequest.url)) {
                return task.httpRequest.url;
            }
            return _this.getUrl(resources, FIREBASE_FUNCTION_URL_FORMAT);
        })
            .then(function (functionUrl) {
            return _this.getServiceAccount()
                .then(function (account) {
                task.httpRequest.oidcToken.serviceAccountEmail = account;
                task.httpRequest.url = functionUrl;
                return task;
            });
        });
    };
    FunctionsApiClient.prototype.toFirebaseError = function (err) {
        if (err instanceof error_1.PrefixedFirebaseError) {
            return err;
        }
        var response = err.response;
        if (!response.isJson()) {
            return new FirebaseFunctionsError('unknown-error', "Unexpected response with status: " + response.status + " and body: " + response.text);
        }
        var error = response.data.error || {};
        var code = 'unknown-error';
        if (error.status && error.status in exports.FUNCTIONS_ERROR_CODE_MAPPING) {
            code = exports.FUNCTIONS_ERROR_CODE_MAPPING[error.status];
        }
        var message = error.message || "Unknown server error: " + response.text;
        return new FirebaseFunctionsError(code, message);
    };
    return FunctionsApiClient;
}());
exports.FunctionsApiClient = FunctionsApiClient;
exports.FUNCTIONS_ERROR_CODE_MAPPING = {
    ABORTED: 'aborted',
    INVALID_ARGUMENT: 'invalid-argument',
    INVALID_CREDENTIAL: 'invalid-credential',
    INTERNAL: 'internal-error',
    FAILED_PRECONDITION: 'failed-precondition',
    PERMISSION_DENIED: 'permission-denied',
    UNAUTHENTICATED: 'unauthenticated',
    NOT_FOUND: 'not-found',
    UNKNOWN: 'unknown-error',
};
/**
 * Firebase Functions error code structure. This extends PrefixedFirebaseError.
 *
 * @param code - The error code.
 * @param message - The error message.
 * @constructor
 */
var FirebaseFunctionsError = /** @class */ (function (_super) {
    __extends(FirebaseFunctionsError, _super);
    function FirebaseFunctionsError(code, message) {
        var _this = _super.call(this, 'functions', code, message) || this;
        /* tslint:disable:max-line-length */
        // Set the prototype explicitly. See the following link for more details:
        // https://github.com/Microsoft/TypeScript/wiki/Breaking-Changes#extending-built-ins-like-error-array-and-map-may-no-longer-work
        /* tslint:enable:max-line-length */
        _this.__proto__ = FirebaseFunctionsError.prototype;
        return _this;
    }
    return FirebaseFunctionsError;
}(error_1.PrefixedFirebaseError));
exports.FirebaseFunctionsError = FirebaseFunctionsError;
